﻿namespace Microsoft.Samples.PlanMyNight.Data.Caching.Tests
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using Microsoft.Samples.PlanMyNight.Data;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Moq;

    [TestClass]
    public class CachedItinerariesRepositoryFixture
    {
        [TestMethod]
        public void SearchByActivityCallsRepository()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.SearchByActivity("NYXXXXX", 5, 1)).Returns(new PagingResult<Itinerary>());

            var cachedRepo = new CachedItinerariesRepository(new Mock<ICachingProvider>().Object, repo.Object);
            var result = cachedRepo.SearchByActivity("NYXXXXX", 5, 1);

            Assert.IsNotNull(result);
            repo.VerifyAll();
        }

        [TestMethod]
        public void SearchByZipCodeCallsRepository()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.SearchByZipCode(1, "10001", 5, 1)).Returns(new PagingResult<Itinerary>());

            var cachedRepo = new CachedItinerariesRepository(new Mock<ICachingProvider>().Object, repo.Object);
            var result = cachedRepo.SearchByZipCode(1, "10001", 5, 1);

            Assert.IsNotNull(result);
            repo.VerifyAll();
        }

        [TestMethod]
        public void SearchByCityCallsRepository()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.SearchByCity(1, "CA", "Colville", 5, 1)).Returns(new PagingResult<Itinerary>());

            var cachedRepo = new CachedItinerariesRepository(new Mock<ICachingProvider>().Object, repo.Object);
            var result = cachedRepo.SearchByCity(1, "CA", "Colville", 5, 1);

            Assert.IsNotNull(result);
            repo.VerifyAll();
        }

        [TestMethod]
        public void SearchByRadiusCallsRepository()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.SearchByRadius(1, 50, 50, 10, 5, 1)).Returns(new PagingResult<Itinerary>());

            var cachedRepo = new CachedItinerariesRepository(new Mock<ICachingProvider>().Object, repo.Object);
            var result = cachedRepo.SearchByRadius(1, 50, 50, 10, 5, 1);

            Assert.IsNotNull(result);
            repo.VerifyAll();
        }

        [TestMethod]
        public void RetrievePrimedCache()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Retrieve(123L)).Returns(new Itinerary { Id = 123L });

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Itinerary", "123")).Returns(null);
            cacheProvider.Setup(o => o.Add("Itinerary", "123", It.IsAny<Itinerary>(), It.IsAny<TimeSpan>()));

            var cachedRepo = new CachedItinerariesRepository(cacheProvider.Object, repo.Object);
            var result = cachedRepo.Retrieve(123L);

            Assert.IsNotNull(result);
            repo.VerifyAll();
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void RetrieveFromCache()
        {
            var repo = new Mock<IItinerariesRepository>();

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Itinerary", "123")).Returns(new Itinerary { Id = 123L });

            var cachedRepo = new CachedItinerariesRepository(cacheProvider.Object, repo.Object);
            var result = cachedRepo.Retrieve(123L);

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void RetrieveByUserCallsRepository()
        {
            var userId = Guid.NewGuid();

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.RetrieveByUser(userId)).Returns(new Itinerary[0]);

            var cachedRepo = new CachedItinerariesRepository(new Mock<ICachingProvider>().Object, repo.Object);
            var result = cachedRepo.RetrieveByUser(userId);

            Assert.IsNotNull(result);
            repo.VerifyAll();
        }

        [TestMethod]
        public void AddCallsRepository()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Add(It.IsAny<Itinerary>()));

            var cachedRepo = new CachedItinerariesRepository(new Mock<ICachingProvider>().Object, repo.Object);
            cachedRepo.Add(new Itinerary { Id = 128, UserId = Guid.NewGuid() });

            repo.VerifyAll();
        }

        [TestMethod]
        public void UpdateCallsRepositoryAndRemovesCacheEntry()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.Update(It.IsAny<Itinerary>()));

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Remove("Itinerary", "123"));

            var cachedRepo = new CachedItinerariesRepository(cacheProvider.Object, repo.Object);
            cachedRepo.Update(new Itinerary { Id = 123, UserId = Guid.NewGuid() });

            repo.VerifyAll();
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void GetUserDisplayNameFromPrimedCache()
        {
            var userId = Guid.NewGuid();

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.GetUserDisplayName(userId)).Returns("John Doe");

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Reference", "username_" + userId.ToString())).Returns(null);
            cacheProvider.Setup(o => o.Add("Reference", "username_" + userId.ToString(), "John Doe", It.IsAny<TimeSpan>()));

            var cachedRepo = new CachedItinerariesRepository(cacheProvider.Object, repo.Object);
            var result = cachedRepo.GetUserDisplayName(userId);

            Assert.AreEqual("John Doe", result);
            repo.VerifyAll();
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void GetUserDisplayNameFromCache()
        {
            var userId = Guid.NewGuid();

            var repo = new Mock<IItinerariesRepository>();

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("Reference", "username_" + userId.ToString())).Returns("John Doe");

            var cachedRepo = new CachedItinerariesRepository(cacheProvider.Object, repo.Object);
            var result = cachedRepo.GetUserDisplayName(userId);

            Assert.AreEqual("John Doe", result);
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void CanUserRateItineraryCallsRepository()
        {
            var userId = Guid.NewGuid();

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.CanUserRateItinerary(123, userId)).Returns(true);

            var cachedRepo = new CachedItinerariesRepository(new Mock<ICachingProvider>().Object, repo.Object);
            var result = cachedRepo.CanUserRateItinerary(123, userId);

            Assert.IsTrue(result);
            repo.VerifyAll();
        }

        [TestMethod]
        public void RateItineraryInvalidatesCache()
        {
            var userId = Guid.NewGuid();

            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.RateItinerary(123, userId, 5, It.IsAny<DateTime>()));

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Remove("Itinerary", "123"));

            var cachedRepo = new CachedItinerariesRepository(cacheProvider.Object, repo.Object);
            cachedRepo.RateItinerary(123, userId, 5, DateTime.Now);

            repo.VerifyAll();
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void GetCommentsFromPrimedCache()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.RetrieveComments(123)).Returns(new ItineraryComment[0]);

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("ItineraryComments", "123")).Returns(null);
            cacheProvider.Setup(o => o.Add("ItineraryComments", "123", It.IsAny<IEnumerable<ItineraryComment>>(), It.IsAny<TimeSpan>()));

            var cachedRepo = new CachedItinerariesRepository(cacheProvider.Object, repo.Object);
            var result = cachedRepo.RetrieveComments(123);

            Assert.IsNotNull(result);
            repo.VerifyAll();
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void GetCommentsFromCache()
        {
            var repo = new Mock<IItinerariesRepository>();

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("ItineraryComments", "123")).Returns(new ItineraryComment[0]);

            var cachedRepo = new CachedItinerariesRepository(cacheProvider.Object, repo.Object);
            var result = cachedRepo.RetrieveComments(123);

            Assert.IsNotNull(result);
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void AddCommentCallsRepository()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.AddComment(It.IsAny<ItineraryComment>()));

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("ItineraryComments", "123")).Returns(null);

            var cachedRepo = new CachedItinerariesRepository(cacheProvider.Object, repo.Object);
            cachedRepo.AddComment(new ItineraryComment { ItineraryId = 123 });

            repo.VerifyAll();
            cacheProvider.VerifyAll();
        }

        [TestMethod]
        public void AddCommentCallsRepositoryAndUpdatesCacheIfFound()
        {
            var repo = new Mock<IItinerariesRepository>();
            repo.Setup(o => o.AddComment(It.IsAny<ItineraryComment>()));

            var cacheProvider = new Mock<ICachingProvider>();
            cacheProvider.Setup(o => o.Get("ItineraryComments", "123")).Returns(new ItineraryComment[0]);
            cacheProvider.Setup(o => o.Add("ItineraryComments", "123", It.Is<IEnumerable<ItineraryComment>>(i => i.Count() == 1), It.IsAny<TimeSpan>()));

            var cachedRepo = new CachedItinerariesRepository(cacheProvider.Object, repo.Object);
            cachedRepo.AddComment(new ItineraryComment { ItineraryId = 123 });

            repo.VerifyAll();
            cacheProvider.VerifyAll();
        }
    }
}
